<?php

namespace App\Http\Controllers\Admin\Keuangan;

use ZipArchive;
use App\Models\Guru;
use App\Models\Pegawai;
use App\Models\Pelatihan;
use App\Models\SpdRincian;
use App\Support\Terbilang;
use Illuminate\Http\Request;
use App\Exports\SpdRekapExport;
use Barryvdh\DomPDF\Facade\Pdf;
use App\Imports\SpdRincianImport;
use App\Http\Controllers\Controller;
use Maatwebsite\Excel\Facades\Excel;

class SpdRincianController extends Controller
{
    public function index(Request $request)
    {
        $search = $request->search;
        $tahun  = $request->tahun;

        $query = SpdRincian::with(['ptk', 'pelatihan'])->latest();

        if ($search) {
            $query->whereHasMorph(
                'ptk',
                [Guru::class, Pegawai::class],
                function ($q) use ($search) {
                    $q->where('nama_guru', 'like', "%{$search}%")
                        ->orWhere('nama', 'like', "%{$search}%")
                        ->orWhere('nip', 'like', "%{$search}%")
                        ->orWhere('nuptk', 'like', "%{$search}%");
                }
            );
        }

        if ($tahun) {
            $query->where('tahun_anggaran', $tahun);
        }

        $rows = $query->paginate(15)->withQueryString();

        return view('admin.keuangan.spd.index', compact('rows', 'search', 'tahun'));
    }

    public function create()
    {
        $pelatihanList = Pelatihan::orderBy('nama_pelatihan')
            ->get(['id', 'nama_pelatihan']);

        return view('admin.keuangan.spd.create', compact('pelatihanList'));
    }


    public function store(Request $request)
    {
        $data = $request->validate([
            'ptk' => ['required', 'string'],
            'pelatihan_id'  => ['nullable', 'exists:pelatihan,id'],
            'tahun_anggaran' => ['required', 'string', 'max:4'],
            'bukti_kas_no'  => ['nullable', 'string', 'max:100'],
            'beban_mak'     => ['nullable', 'string', 'max:150'],
            'lampiran_spd'  => ['nullable', 'string', 'max:150'],
            'nomor_spd'     => ['nullable', 'string', 'max:150'],
            'tanggal_spd'   => ['nullable', 'date'],

            'uraian'        => ['nullable', 'array', 'min:1'],
            'uraian.*'      => ['nullable', 'string'],
            'qty'           => ['nullable', 'array'],
            'qty.*'         => ['nullable', 'numeric', 'min:1'],
            'harga_satuan'  => ['nullable', 'array'],
            'harga_satuan.*' => ['nullable', 'numeric', 'min:0'],
            'keterangan_item' => ['nullable', 'array'],
        ]);

        /* ================= PTK VALIDATION ================= */

        if (!str_contains($data['ptk'], '_')) {
            return back()
                ->withInput()
                ->with('error', 'Format PTK tidak valid.');
        }

        [$ptkPrefix, $ptkId] = explode('_', $data['ptk']);

        $ptkType = match ($ptkPrefix) {
            'guru'    => Guru::class,
            'pegawai' => Pegawai::class,
            default   => null,
        };

        if (!$ptkType || !is_numeric($ptkId)) {
            return back()
                ->withInput()
                ->with('error', 'PTK tidak dikenali.');
        }

        /* ================= ITEMS ================= */

        $items = [];
        $total = 0;

        foreach ($request->uraian as $i => $uraian) {
            if (empty($uraian)) {
                continue;
            }

            $qty   = (float) ($request->qty[$i] ?? 0);
            $harga = (float) ($request->harga_satuan[$i] ?? 0);
            $jumlah = (int) round($qty * $harga);

            $items[] = [
                'uraian'       => $uraian,
                'qty'          => $qty,
                'harga_satuan' => (int) round($harga),
                'jumlah'       => $jumlah,
                'keterangan'   => $request->keterangan_item[$i] ?? null,
            ];

            $total += $jumlah;
        }

        if (empty($items)) {
            return back()
                ->withInput()
                ->with('error', 'Minimal harus ada satu baris rincian biaya.');
        }

        /* ================= SAVE ================= */

        $spd = SpdRincian::create([
            'ptk_type'       => $ptkType,
            'ptk_id'         => $ptkId,
            'pelatihan_id'   => $data['pelatihan_id'] ?? null,
            'tahun_anggaran' => $data['tahun_anggaran'],
            'bukti_kas_no'   => $data['bukti_kas_no'] ?: null,
            'beban_mak'      => $data['beban_mak'] ?: null,
            'lampiran_spd'   => $data['lampiran_spd'] ?: null,
            'nomor_spd'      => $data['nomor_spd'] ?: null,
            'tanggal_spd'    => $data['tanggal_spd'] ?: null,
            'items'          => $items,
            'total'          => $total,
            'created_by'     => auth()->id(),
        ]);

        return redirect()
            ->route('admin.keuangan.spd.show', $spd->id)
            ->with('success', 'Rincian SPD berhasil dibuat.');
    }


    public function show(SpdRincian $spd)
    {
        $spd->load(['ptk', 'pelatihan']);
        $terbilang = trim(Terbilang::make($spd->total));

        return view('admin.keuangan.spd.show', compact('spd', 'terbilang'));
    }

    public function print(SpdRincian $spd)
    {
        $spd->load(['ptk', 'pelatihan']);
        $terbilang = trim(Terbilang::make($spd->total));

        // === KOP SURAT (cek jika ada file lokal di /public/images/kop_bgtk_revisi.jpeg) ===
        $kopPath = public_path('images/kop_bgtk_revisi.jpeg');
        $kop = null;

        if (file_exists($kopPath)) {
            $kop = asset('images/kop_bgtk_revisi.jpeg');
        }

        return view('admin.keuangan.spd.print', compact('spd', 'terbilang', 'kop'));
    }


    public function download(SpdRincian $spd)
    {
        $spd->load(['pelatihan']);

        $terbilang = trim(Terbilang::make($spd->total));

        // === KOP untuk PDF ===
        $kopPath = public_path('images/kop_bgtk_terbaru.png');
        $kop = file_exists($kopPath) ? $kopPath : null;

        // 🔥 Nama PTK aman (snapshot)
        $namaPtk = $spd->nama_ptk ?? 'ptk';

        // Biar aman buat nama file
        $namaFilePtk = str_replace(
            ['/', '\\', ':', '*', '?', '"', '<', '>', '|'],
            '',
            $namaPtk
        );

        $pdf = Pdf::loadView('admin.keuangan.spd.print', [
            'spd'       => $spd,
            'terbilang' => $terbilang,
            'kop'       => $kop,
        ])->setPaper('a4');

        return $pdf->download(
            'rincian_spd_' . $namaFilePtk . '.pdf'
        );
    }


    public function downloadAll()
    {
        // 🔥 Naikkan limit (AMAN untuk proses berat)
        ini_set('max_execution_time', 300); // 5 menit
        ini_set('memory_limit', '512M');

        $totalData = SpdRincian::count();

        if ($totalData === 0) {
            return back()->with('error', 'Tidak ada data untuk diunduh.');
        }

        $tempDir = storage_path('app/temp_spd_pdf/');
        if (!file_exists($tempDir)) {
            mkdir($tempDir, 0777, true);
        }

        // bersihkan file lama
        foreach (glob($tempDir . '*.pdf') as $oldFile) {
            @unlink($oldFile);
        }

        // 🔥 KOP LOAD SEKALI
        $kopPath = public_path('images/kop_bgtk_terbaru.png');
        $kop = file_exists($kopPath) ? $kopPath : null;

        // 🔥 PROSES BERTAHAP (ANTI RAM JEBOL)
        SpdRincian::with(['ptk', 'pelatihan'])
            ->orderBy('id')
            ->chunk(10, function ($rows) use ($tempDir, $kop) {

                foreach ($rows as $spd) {

                    $terbilang = trim(\App\Support\Terbilang::make($spd->total));

                    $rawName = $spd->ptk?->nama_guru
                        ?? $spd->ptk?->nama
                        ?? $spd->nama_ptk
                        ?? 'ptk';

                    $safeName = preg_replace('/[^A-Za-z0-9_\-]/', '_', $rawName);
                    $filename = 'spd_' . $safeName . '_' . $spd->id . '.pdf';

                    Pdf::loadView('admin.keuangan.spd.print', [
                        'spd'       => $spd,
                        'terbilang' => $terbilang,
                        'kop'       => $kop,
                    ])
                        ->setPaper('a4')
                        ->setOptions([
                            'isRemoteEnabled' => true,
                            'dpi' => 96,
                            'defaultFont' => 'Times-Roman',
                        ])
                        ->save($tempDir . $filename);
                }
            });

        $pdfFiles = glob($tempDir . '*.pdf');

        if (count($pdfFiles) === 0) {
            return back()->with('error', 'PDF tidak berhasil dibuat.');
        }

        // 🔥 ZIP
        $zipPath = storage_path('app/rincian_spd_all.zip');
        $zip = new \ZipArchive;

        if ($zip->open($zipPath, \ZipArchive::CREATE | \ZipArchive::OVERWRITE)) {
            foreach ($pdfFiles as $file) {
                $zip->addFile($file, basename($file));
            }
            $zip->close();
        }

        return response()->download($zipPath)->deleteFileAfterSend(true);
    }



    public function downloadExcel()
    {
        return Excel::download(
            new SpdRekapExport,
            'rekap_spd_' . now()->format('Ymd_His') . '.xlsx'
        );
    }

    public function destroy(SpdRincian $spd)
    {
        $spd->delete();

        return redirect()
            ->route('admin.keuangan.spd.index')
            ->with('success', 'SPD berhasil dihapus.');
    }

    public function import(Request $request)
    {
        $request->validate([
            'file' => ['required', 'file', 'mimes:xlsx,csv'],
        ]);

        Excel::import(new SpdRincianImport, $request->file('file'));

        return back()->with('success', 'Import SPD berhasil.');
    }

    public function edit(SpdRincian $spd)
    {
        $spd->load(['ptk', 'pelatihan']);

        $pelatihanList = Pelatihan::orderBy('nama_pelatihan')
            ->get(['id', 'nama_pelatihan']);

        // value select2
        $selectedPTK = null;
        if ($spd->ptk) {
            $prefix = $spd->ptk_type === Guru::class ? 'guru' : 'pegawai';
            $nama   = $spd->ptk->nama_guru ?? $spd->ptk->nama;
            $nip    = $spd->ptk->nip ?? null;

            $selectedPTK = [
                'id'   => "{$prefix}_{$spd->ptk_id}",
                'text' => $nama . ($nip ? " - ({$nip})" : ''),
            ];
        }

        return view('admin.keuangan.spd.edit', compact(
            'spd',
            'pelatihanList',
            'selectedPTK'
        ));
    }

    public function update(Request $request, SpdRincian $spd)
    {
        $data = $request->validate([
            'ptk'             => 'required|string', // cuma buat lolos validasi
            'pelatihan_id'    => 'nullable|exists:pelatihan,id',
            'tahun_anggaran'  => 'required|string|max:4',
            'bukti_kas_no'    => 'nullable|string|max:100',
            'beban_mak'       => 'nullable|string|max:150',
            'lampiran_spd'    => 'nullable|string|max:150',
            'nomor_spd'       => 'nullable|string|max:150',
            'tanggal_spd'     => 'nullable|date',

            'uraian'          => 'required|array|min:1',
            'uraian.*'        => 'nullable|string',
            'qty'             => 'nullable|array',
            'qty.*'           => 'nullable|numeric|min:1',
            'harga_satuan'    => 'nullable|array',
            'harga_satuan.*'  => 'nullable|numeric|min:0',
            'keterangan_item' => 'nullable|array',
        ]);

        /* ===== PTK (LOCKED) ===== */
        $ptkType = $spd->ptk_type;
        $ptkId   = $spd->ptk_id;

        /* ===== ITEMS ===== */
        $items = [];
        $total = 0;

        foreach ($request->uraian as $i => $uraian) {
            if (!$uraian) continue;

            $qty   = (float) ($request->qty[$i] ?? 0);
            $harga = (float) ($request->harga_satuan[$i] ?? 0);
            $jumlah = (int) round($qty * $harga);

            $items[] = [
                'uraian'       => $uraian,
                'qty'          => $qty,
                'harga_satuan' => (int) round($harga),
                'jumlah'       => $jumlah,
                'keterangan'   => $request->keterangan_item[$i] ?? null,
            ];

            $total += $jumlah;
        }

        if (!$items) {
            return back()->withInput()->with('error', 'Minimal satu rincian biaya harus diisi.');
        }

        /* ===== UPDATE ===== */
        $spd->update([
            'ptk_type'       => $ptkType,
            'ptk_id'         => $ptkId,
            'pelatihan_id'   => $data['pelatihan_id'],
            'tahun_anggaran' => $data['tahun_anggaran'],
            'bukti_kas_no'   => $data['bukti_kas_no'],
            'beban_mak'      => $data['beban_mak'],
            'lampiran_spd'   => $data['lampiran_spd'],
            'nomor_spd'      => $data['nomor_spd'],
            'tanggal_spd'    => $data['tanggal_spd'],
            'items'          => $items,
            'total'          => $total,
            'updated_by'     => auth()->id(),
        ]);

        return redirect()
            ->route('admin.keuangan.spd.show', $spd->id)
            ->with('success', 'Rincian SPD berhasil diperbarui.');
    }
}
