<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Pelatihan;
use App\Models\PelatihanKelasJadwal;
use App\Models\RegistrasiPeserta;
use Carbon\Carbon;

class LaporanKegiatanController extends Controller
{
    // halaman list pelatihan -> kelas -> jadwal (biar admin tinggal klik download)
    public function index()
    {
        $pelatihan = Pelatihan::with([
            'kelas' => function ($q) {
                $q->orderBy('nama_kelas');
            },
            'kelas.jadwal' => function ($q) {
                // ini akan pakai FK dari relasi model jadwal() yang sudah kita betulkan
                $q->orderBy('tanggal')
                  ->orderBy('jam_mulai');
            }
        ])->orderBy('nama_pelatihan')->get();

        return view('admin.laporan_kegiatan.index', compact('pelatihan'));
    }

    // Download laporan kegiatan (DOC)
    public function downloadLaporan(PelatihanKelasJadwal $jadwal)
    {
        $jadwal->load(['kelas.pelatihan']);

        $kelas = $jadwal->kelas;
        $pelatihan = $kelas->pelatihan;

        $jumlahPeserta = RegistrasiPeserta::where('pelatihan_id', $pelatihan->id)->count();

        $hari = $jadwal->tanggal ? Carbon::parse($jadwal->tanggal)->translatedFormat('l') : '-';
        $tgl  = $jadwal->tanggal ? Carbon::parse($jadwal->tanggal)->translatedFormat('d F Y') : '-';

        $html = view('admin.laporan_kegiatan.doc_laporan', compact(
            'pelatihan', 'kelas', 'jadwal', 'jumlahPeserta', 'hari', 'tgl'
        ))->render();

        $filename = $this->safeFilename('Laporan-Kegiatan-' . ($pelatihan->nama_pelatihan ?? 'Pelatihan') . '.doc');

        return response($html)
            ->header('Content-Type', 'application/msword')
            ->header('Content-Disposition', 'attachment; filename="'.$filename.'"');
    }

    // Download daftar hadir peserta (DOC)
    public function downloadDaftarHadirPeserta(PelatihanKelasJadwal $jadwal)
    {
        $jadwal->load(['kelas.pelatihan']);
        $kelas = $jadwal->kelas;
        $pelatihan = $kelas->pelatihan;

        $peserta = RegistrasiPeserta::where('pelatihan_id', $pelatihan->id)
            ->orderBy('nama')
            ->get();

        $hari = $jadwal->tanggal ? Carbon::parse($jadwal->tanggal)->translatedFormat('l') : '-';
        $tgl  = $jadwal->tanggal ? Carbon::parse($jadwal->tanggal)->translatedFormat('d F Y') : '-';

        $html = view('admin.laporan_kegiatan.doc_absensi_peserta', compact(
            'pelatihan', 'kelas', 'jadwal', 'peserta', 'hari', 'tgl'
        ))->render();

        $filename = $this->safeFilename('Daftar-Hadir-Peserta-' . ($pelatihan->nama_pelatihan ?? 'Pelatihan') . '.doc');

        return response($html)
            ->header('Content-Type', 'application/msword')
            ->header('Content-Disposition', 'attachment; filename="'.$filename.'"');
    }

    // Download daftar hadir fasil (DOC)
    public function downloadDaftarHadirFasil(PelatihanKelasJadwal $jadwal)
    {
        $jadwal->load([
            'kelas.pelatihan',
            'absensiFasil' => function ($q) {
                $q->orderBy('nama_fasil');
            }
        ]);

        $kelas = $jadwal->kelas;
        $pelatihan = $kelas->pelatihan;

        $hari = $jadwal->tanggal ? Carbon::parse($jadwal->tanggal)->translatedFormat('l') : '-';
        $tgl  = $jadwal->tanggal ? Carbon::parse($jadwal->tanggal)->translatedFormat('d F Y') : '-';

        $fasil = $jadwal->absensiFasil->map(function ($row) use ($jadwal) {

            // amanin format jam: bisa "H:i", "H:i:s", atau null
            $mulaiStr   = $row->jam_mulai ?? $jadwal->jam_mulai;
            $selesaiStr = $row->jam_selesai ?? $jadwal->jam_selesai;

            $mulai   = $this->parseTime($mulaiStr);
            $selesai = $this->parseTime($selesaiStr);

            $durasiMenit = 0;
            if ($mulai && $selesai) {
                $durasiMenit = $selesai->diffInMinutes($mulai);
            }

            // kurangi ishoma jika ada dan valid
            if ($jadwal->ishoma_mulai && $jadwal->ishoma_selesai) {
                $iMulai   = $this->parseTime($jadwal->ishoma_mulai);
                $iSelesai = $this->parseTime($jadwal->ishoma_selesai);

                if ($iMulai && $iSelesai) {
                    $durasiMenit = max(0, $durasiMenit - $iSelesai->diffInMinutes($iMulai));
                }
            }

            $jp = (int) ceil($durasiMenit / 45);

            $row->durasi_menit = $durasiMenit;
            $row->jp = $jp;

            return $row;
        });

        $html = view('admin.laporan_kegiatan.doc_absensi_fasil', compact(
            'pelatihan', 'kelas', 'jadwal', 'fasil', 'hari', 'tgl'
        ))->render();

        $filename = $this->safeFilename('Daftar-Hadir-Fasil-' . ($pelatihan->nama_pelatihan ?? 'Pelatihan') . '.doc');

        return response($html)
            ->header('Content-Type', 'application/msword')
            ->header('Content-Disposition', 'attachment; filename="'.$filename.'"');
    }

    private function safeFilename(string $name): string
    {
        // buang karakter aneh biar aman di Windows
        $name = preg_replace('/[^\w\-\. ]+/u', '', $name);
        $name = str_replace(' ', '-', $name);
        return $name;
    }

    private function parseTime($time)
    {
        if (!$time) return null;

        // coba format H:i:s dulu, kalau gagal coba H:i
        try {
            return Carbon::createFromFormat('H:i:s', $time);
        } catch (\Throwable $e) {
            try {
                return Carbon::createFromFormat('H:i', $time);
            } catch (\Throwable $e2) {
                return null;
            }
        }
    }
}
