<?php

namespace App\Http\Controllers;

use App\Models\DaftarHadir;
use App\Models\SuratTugas;
use Illuminate\Http\Request;
use Barryvdh\DomPDF\Facade\Pdf;

class DaftarHadirController extends Controller
{
    public function index()
    {
        $data = DaftarHadir::with([
            'suratTugas',
            'detail'
        ])->latest()->get();

        return view(
            'admin.kegiatan.daftar-hadir.index',
            compact('data')
        );
    }

    public function create()
    {
        $suratTugas = SuratTugas::withCount('pegawais')->get();

        return view(
            'admin.kegiatan.daftar-hadir.create',
            compact('suratTugas')
        );
    }

    public function store(Request $request)
    {
        $request->validate([
            'surat_tugas_id' => 'required|exists:surat_tugas,id',
            'tanggal'        => 'required|date',
            'tempat'         => 'required|string|max:255',
            'nama_manual.*'  => 'nullable|string|max:255',
        ]);

        // Cegah dobel daftar hadir
        if (DaftarHadir::where('surat_tugas_id', $request->surat_tugas_id)->exists()) {
            return back()->withErrors([
                'surat_tugas_id' => 'Daftar hadir untuk surat tugas ini sudah ada'
            ]);
        }

        $daftarHadir = DaftarHadir::create([
            'surat_tugas_id' => $request->surat_tugas_id,
            'tanggal'        => $request->tanggal,
            'tempat'         => $request->tempat,
        ]);

        /**
         * PESERTA OTOMATIS DARI SURAT TUGAS
         */
        $surat = SuratTugas::with('pegawais')->find($request->surat_tugas_id);

        foreach ($surat->pegawais as $pegawai) {
            $daftarHadir->detail()->create([
                'pegawai_id' => $pegawai->id,
                'tipe'       => 'peserta'
            ]);
        }

        /**
         * PESERTA MANUAL (JIKA ADA)
         */
        if ($request->nama_manual) {
            foreach ($request->nama_manual as $nama) {
                if ($nama) {
                    $daftarHadir->detail()->create([
                        'nama_manual' => $nama,
                        'tipe'        => 'peserta'
                    ]);
                }
            }
        }

        return redirect()
            ->route('daftar-hadir.index')
            ->with('success', 'Daftar hadir berhasil dibuat');
    }

    public function edit($id)
    {
        $daftarHadir = DaftarHadir::with([
            'suratTugas',
            'detail' => function ($q) {
                $q->where('tipe', 'peserta');
            }
        ])->findOrFail($id);

        return view(
            'admin.kegiatan.daftar-hadir.edit',
            compact('daftarHadir')
        );
    }

    public function update(Request $request, $id)
    {
        $request->validate([
            'tanggal' => 'required|date',
            'tempat'  => 'required|string|max:255',
            'manual_existing.*' => 'nullable|string|max:255',
            'manual_new.*'      => 'nullable|string|max:255',
        ]);

        $daftarHadir = DaftarHadir::with('detail')->findOrFail($id);

        // update header
        $daftarHadir->update([
            'tanggal' => $request->tanggal,
            'tempat'  => $request->tempat,
        ]);

        /* ============================
        * UPDATE PESERTA MANUAL LAMA
        * ============================ */
        $existingIds = [];

        if ($request->manual_existing) {
            foreach ($request->manual_existing as $detailId => $nama) {
                if ($nama) {
                    $daftarHadir->detail()
                        ->where('id', $detailId)
                        ->update([
                            'nama_manual' => $nama,
                            'tipe' => 'peserta',
                            'pegawai_id' => null,
                        ]);

                    $existingIds[] = $detailId;
                }
            }
        }

        // hapus manual yang dihapus dari form
        $daftarHadir->detail()
            ->where('tipe','peserta')
            ->whereNull('pegawai_id')
            ->whereNotIn('id', $existingIds)
            ->delete();

        /* ============================
        * TAMBAH PESERTA MANUAL BARU
        * ============================ */
        if ($request->manual_new) {
            foreach ($request->manual_new as $nama) {
                if ($nama) {
                    $daftarHadir->detail()->create([
                        'nama_manual' => $nama,
                        'tipe' => 'peserta',
                        'pegawai_id' => null,
                    ]);
                }
            }
        }

        return redirect()
            ->route('daftar-hadir.index')
            ->with('success','Daftar hadir peserta berhasil diperbarui');
    }

    public function destroy($id)
    {
        $daftarHadir = DaftarHadir::findOrFail($id);

        $daftarHadir->detail()->delete();
        $daftarHadir->delete();

        return back()->with('success', 'Daftar hadir dihapus');
    }

    public function cetak($id)
    {
        $daftarHadir = DaftarHadir::with([
            'suratTugas',
            'detail.pegawai'
        ])->findOrFail($id);

        return Pdf::loadView(
            'admin.kegiatan.pdf.daftar-hadir',
            compact('daftarHadir')
        )->stream('daftar-hadir.pdf');
    }

}
