<?php

namespace App\Http\Controllers;

use App\Models\Pegawai;
use App\Models\Sekolah;
use App\Exports\PegawaiExport;
use App\Imports\PegawaiImport;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Maatwebsite\Excel\Facades\Excel;

class PegawaiController extends Controller
{
    public function index(Request $request)
    {
        $search = $request->input('search');

        $query = Pegawai::orderBy('nama');

        if ($search) {
            $query->where('nama', 'like', "%{$search}%")
                  ->orWhere('nip', 'like', "%{$search}%")
                  ->orWhere('jabatan', 'like', "%{$search}%");
        }

        $pegawai = $query->paginate(20)->withQueryString();

        return view('admin.pegawai.index', compact(
            'pegawai',
            'search'
        ));
    }

    public function create()
    {
        return view('admin.pegawai.create');
    }

    public function edit(Pegawai $pegawai)
    {
        return view('admin.pegawai.edit', compact('pegawai'));
    }

    public function store(Request $request)
    {
        $data = $this->validateData($request);

        if ($request->hasFile('foto')) {
            $data['foto'] = $request->file('foto')->store('pegawai', 'public');
        }

        Pegawai::create($data);

        return redirect()->route('pegawai.index')->with('success', 'Data pegawai berhasil ditambahkan.');
    }

    public function update(Request $request, Pegawai $pegawai)
    {
        $data = $this->validateData($request);

        if ($request->hasFile('foto')) {
            if ($pegawai->foto && Storage::disk('public')->exists($pegawai->foto)) {
                Storage::disk('public')->delete($pegawai->foto);
            }
            $data['foto'] = $request->file('foto')->store('pegawai', 'public');
        }

        $pegawai->update($data);

        return redirect()->route('pegawai.index')->with('success', 'Data pegawai berhasil diperbarui.');
    }

    protected function validateData(Request $request)
    {
        return $request->validate([
            'nama'               => 'required|string|max:255',
            'nip'                => 'nullable|string|max:50',
            'gol'                => 'nullable|string|max:50',
            'pangkat_golongan'   => 'nullable|string|max:50',
            'jabatan'            => 'nullable|string|max:255',
            'status_kepegawaian' => 'nullable|string|max:50',
            'instansi'            => 'nullable|string|max:255',
            'foto'               => 'nullable|image|mimes:jpg,jpeg,png|max:2048',
            'email'              => 'nullable|email|max:255',
            'telepon'            => 'nullable|string|max:50',
            'alamat'             => 'nullable|string',
            'tanggal_lahir'      => 'nullable|date',
            'jenis_kelamin'      => 'nullable|in:Laki-laki,Perempuan',
        ]);
    }

    public function destroy(Pegawai $pegawai)
    {
        if ($pegawai->foto && Storage::disk('public')->exists($pegawai->foto)) {
            Storage::disk('public')->delete($pegawai->foto);
        }

        $pegawai->delete();

        return redirect()->route('pegawai.index')->with('success', 'Data pegawai berhasil dihapus.');
    }

    public function show(Pegawai $pegawai)
    {
        return view('admin.pegawai.show', compact('pegawai'));
    }

    public function import(Request $request)
    {
        $request->validate([
            'file' => 'required|mimes:xlsx,xls|max:5120',
        ]);

        try {
            Excel::import(new PegawaiImport, $request->file('file'));

            return redirect()->route('pegawai.index')->with('success', 'Import data pegawai berhasil.');
        } catch (\Throwable $e) {
            return back()->with('error', 'Gagal import: ' . $e->getMessage());
        }
    }

    public function export()
    {
        $fileName = 'data_pegawai_' . now()->format('Ymd_His') . '.xlsx';
        return Excel::download(new PegawaiExport, $fileName);
    }

    public function select2(Request $request)
    {
        $search = $request->input('q');

        $query = Pegawai::select('id', 'nama', 'nip', 'jabatan')->orderBy('nama');

        if ($search) {
            $query->where('nama', 'like', "%{$search}%")
                  ->orWhere('nip', 'like', "%{$search}%")
                  ->orWhere('jabatan', 'like', "%{$search}%");
        }

        $data = $query->limit(30)->get();

        $result = $data->map(function ($p) {
            $identitas = $p->nip ? " — {$p->nip}" : '';
            $jabatan   = $p->jabatan ? " — {$p->jabatan}" : '';

            return [
                'id'   => $p->id,
                'text' => $p->nama . $identitas . $jabatan,
            ];
        });

        return response()->json(['results' => $result]);
    }
}
