<?php

namespace App\Http\Controllers;

use App\Models\Guru;
use App\Models\Sekolah;
use Illuminate\Http\Request;
use App\Exports\SekolahExport;
use App\Imports\SekolahImport;
use Maatwebsite\Excel\Facades\Excel;

class SekolahController extends Controller
{
    public function index(Request $request)
    {
        $search    = $request->input('search');
        $kabupaten = $request->input('kabupaten');
        $jenjang   = $request->input('jenjang');
        $status    = $request->input('status_sekolah');

        $query = Sekolah::query()->orderBy('nama');

        // Pencarian bebas: nama / NPSN / sekolah_id (dapodik)
        if ($search) {
            $query->where(function ($q) use ($search) {
                $q->where('nama', 'like', "%{$search}%")
                  ->orWhere('npsn', 'like', "%{$search}%")
                  ->orWhere('sekolah_id', 'like', "%{$search}%");
            });
        }

        // Filter kota/kabupaten
        if ($kabupaten) {
            $query->where('kabupaten', $kabupaten);
        }

        // Filter jenjang
        if ($jenjang) {
            $query->where('jenjang', $jenjang);
        }

        // Filter status sekolah (Negeri / Swasta)
        if ($status) {
            $query->where('status_sekolah', $status);
        }

        $sekolah = $query->paginate(20)->withQueryString();

        // Opsi filter di dropdown – ambil langsung dari database supaya pasti matching
        $kabupatenOptions = Sekolah::select('kabupaten')
            ->whereNotNull('kabupaten')
            ->distinct()
            ->orderBy('kabupaten')
            ->pluck('kabupaten')
            ->toArray();

        $jenjangList = Sekolah::select('jenjang')
            ->whereNotNull('jenjang')
            ->distinct()
            ->orderBy('jenjang')
            ->get();

        $statusOptions = Sekolah::select('status_sekolah')
            ->whereNotNull('status_sekolah')
            ->distinct()
            ->orderBy('status_sekolah')
            ->pluck('status_sekolah')
            ->toArray();

        $kabupatenOptions = Sekolah::select('kabupaten')
        ->whereNotNull('kabupaten')
        ->distinct()
        ->orderBy('kabupaten')
        ->pluck('kabupaten');

        return view('admin.sekolah.index', compact(
            'sekolah',
            'search',
            'kabupaten',
            'jenjang',
            'status',
            'kabupatenOptions',
            'jenjangList',
            'statusOptions'
        ));
    }

    public function create()
    {
        return view('admin.sekolah.create');
    }

    public function store(Request $request)
    {
        $data = $request->validate([
            'sekolah_id'       => ['nullable', 'string', 'max:50'],
            'npsn'             => ['nullable', 'string', 'max:50'],
            'nama'             => ['required', 'string', 'max:255'],
            'jenjang'          => ['nullable', 'string', 'max:50'],
            'alamat_jalan'     => ['nullable', 'string'],
            'kecamatan'        => ['nullable', 'string', 'max:100'],
            'kabupaten'        => ['nullable', 'string', 'max:100'],
            'status_sekolah'   => ['nullable', 'string', 'max:50'],
            'nomor_telepon'    => ['nullable', 'string', 'max:50'],
            'email'            => ['nullable', 'string', 'max:255'],
        ]);

        Sekolah::create($data);

        return redirect()->route('sekolah.index')->with('success', 'Data sekolah berhasil ditambahkan.');
    }

    public function show(Sekolah $sekolah)
    {
        $sekolah->load(['guru' => function($q){
            $q->orderBy('nama_guru');
        }]);

        return view('admin.sekolah.show', compact('sekolah'));
    }

    public function edit(Sekolah $sekolah)
    {
        return view('admin.sekolah.edit', compact('sekolah'));
    }

    public function update(Request $request, Sekolah $sekolah)
    {
        $data = $request->validate([
            'sekolah_id'       => ['nullable', 'string', 'max:50'],
            'npsn'             => ['nullable', 'string', 'max:50'],
            'nama'             => ['required', 'string', 'max:255'],
            'jenjang'          => ['nullable', 'string', 'max:50'],
            'alamat_jalan'     => ['nullable', 'string'],
            'kecamatan'        => ['nullable', 'string', 'max:100'],
            'sekolah.kabupaten'        => ['nullable', 'string', 'max:100'],
            'status_sekolah'   => ['nullable', 'string', 'max:50'],
            'nomor_telepon'    => ['nullable', 'string', 'max:50'],
            'email'            => ['nullable', 'string', 'max:255'],
        ]);

        $sekolah->update($data);

        return redirect()->route('sekolah.index')->with('success', 'Data sekolah berhasil diperbarui.');
    }

    public function destroy(Sekolah $sekolah)
    {
        try {
            // Jika sekolah punya guru, hapus relasinya dulu / putus foreign key
            Guru::where('sekolah_id', $sekolah->id)->update(['sekolah_id' => null]);

            $sekolah->delete();

            return redirect()
                ->route('sekolah.index')
                ->with('success', 'Data sekolah berhasil dihapus.');
        } catch (\Throwable $e) {
            return back()->with('error', 'Gagal hapus: ' . $e->getMessage());
        }
    }

    public function import(Request $request)
    {
        $request->validate([
            'file' => ['required', 'file', 'mimes:xlsx,xls', 'max:5120'],
        ]);

        try {
            Excel::import(new SekolahImport, $request->file('file'));

            return redirect()
                ->route('admin.sekolah.index')
                ->with('success', 'Import data sekolah dari file Excel berhasil.');
        } catch (\Throwable $e) {
            return redirect()
                ->route('admin.sekolah.index')
                ->with('error', 'Gagal import data sekolah: ' . $e->getMessage());
        }
    }

    public function export(Request $request)
{
    $filename = 'data_sekolah_bgtk_' . now()->format('Ymd_His') . '.xlsx';
    $filters = $request->only(['search', 'kabupaten', 'jenjang', 'status_sekolah']);
    return Excel::download(new SekolahExport($filters), $filename);
}

}
