<?php

namespace App\Http\Controllers;

use App\Models\SkKegiatan;
use App\Models\Kegiatan;
use App\Models\Pegawai;
use Illuminate\Http\Request;
use Barryvdh\DomPDF\Facade\Pdf;

class SkKegiatanController extends Controller
{
    public function index()
    {
        $data = SkKegiatan::orderBy('tanggal_sk', 'desc')->get();
        return view('admin.kegiatan.sk.index', compact('data'));
    }

    public function create()
    {
        $pegawais = Pegawai::orderBy('jabatan')
            ->orderBy('nama')
            ->get()
            ->groupBy('jabatan');

        $selectedAnggota = []; // ⬅️ penting untuk CREATE

        return view('admin.kegiatan.sk.create', compact('pegawais', 'selectedAnggota'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'nama_kegiatan' => 'required',
            'nomor_sk' => 'required',
            'tanggal_sk' => 'required|date',
            'pejabat_penandatangan' => 'required',
            'jabatan_penandatangan' => 'required',
            'anggota' => 'required|array'
        ]);

        $sk = SkKegiatan::create([
            'nama_kegiatan' => $request->nama_kegiatan,
            'nomor_sk' => $request->nomor_sk,
            'tanggal_sk' => $request->tanggal_sk,
            'pejabat_penandatangan' => $request->pejabat_penandatangan,
            'nip_penandatangan' => $request->nip_penandatangan,
            'jabatan_penandatangan' => $request->jabatan_penandatangan,
        ]);

        foreach ($request->anggota as $pegawaiId) {
            $sk->anggota()->create([
                'pegawai_id' => $pegawaiId,
                'instansi' => 'BGTK PROVINSI DKI JAKARTA',
                'keterangan' => null,
            ]);
        }        

        return redirect()->route('sk.index')
            ->with('success', 'SK berhasil dibuat');
    }

    public function cetak($id)
    {
        $sk = SkKegiatan::with('anggota.pegawai', 'kegiatan')
            ->findOrFail($id);

        $namaFile = 'SK-' . str_replace(['/', '\\'], '-', $sk->nomor_sk) . '.pdf';

        $pdf = Pdf::loadView(
            'admin.kegiatan.pdf.sk',
            compact('sk')
        )->setPaper('A4', 'portrait');

        return $pdf->stream($namaFile);
    }

    public function edit(SkKegiatan $sk)
    {
        $pegawais = Pegawai::orderBy('jabatan')->orderBy('nama')->get()->groupBy('jabatan');
        $selectedAnggota = $sk->anggota->pluck('pegawai_id')->toArray();

        return view('admin.kegiatan.sk.edit', compact(
            'sk',
            'pegawais',
            'selectedAnggota'
        ));
    }

    public function update(Request $request, SkKegiatan $sk)
    {
        $request->validate([
            'nama_kegiatan' => 'required',
            'nomor_sk' => 'required',
            'tanggal_sk' => 'required|date',
            'anggota' => 'required|array'
        ]);

        $sk->update($request->only([
            'nama_kegiatan',
            'nomor_sk',
            'tanggal_sk',
            'pejabat_penandatangan',
            'nip_penandatangan',
            'jabatan_penandatangan',
        ]));

        // 🔄 reset lampiran panitia
        $sk->anggota()->delete();

        foreach ($request->anggota as $pegawaiId) {
            $sk->anggota()->create([
                'pegawai_id' => $pegawaiId,
                'instansi' => 'BGTK PROVINSI DKI JAKARTA',
            ]);
        }

        return redirect()->route('sk.index')
            ->with('success', 'SK berhasil diperbarui');
    }

    public function destroy(SkKegiatan $sk)
    {
        $sk->anggota()->delete();
        $sk->delete();

        return redirect()->route('sk.index')
            ->with('success', 'SK berhasil dihapus');
    }

}
