<?php

namespace App\Imports;

use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;
use Maatwebsite\Excel\Concerns\ToCollection;
use Maatwebsite\Excel\Concerns\WithHeadingRow;
use Maatwebsite\Excel\Concerns\WithChunkReading;
use PhpOffice\PhpSpreadsheet\Shared\Date;

class GuruImport implements ToCollection, WithHeadingRow, WithChunkReading
{
    protected array $sekolahMap;

    public function __construct()
    {
        $this->sekolahMap = DB::table('sekolah')
            ->pluck('id', 'sekolah_id')
            ->toArray();
    }

    public function collection(Collection $rows)
    {
        $batch = [];
        $now   = now();

        foreach ($rows as $row) {
            $row = array_change_key_case($row->toArray(), CASE_LOWER);

            if (
                empty($row['nama']) &&
                empty($row['email']) &&
                empty($row['nip']) &&
                empty($row['nuptk'])
            ) {
                continue;
            }

            // =========================
            // NORMALISASI DATA UNIQUE
            // =========================
            $email = trim(strtolower($row['email'] ?? ''));
            $email = $email === '' ? null : $email;

            $nik = trim($row['nik'] ?? '');
            $nik = $nik === '' ? null : $nik;

            $nip = trim($row['nip'] ?? '');
            $nip = $nip === '' ? null : $nip;

            $nuptk = trim($row['nuptk'] ?? '');
            $nuptk = $nuptk === '' ? null : $nuptk;

            // =========================
            // SEKOLAH ID
            // =========================
            $dapodikId = $row['sekolah_id'] ?? null;
            $sekolahId = $this->sekolahMap[$dapodikId] ?? null;

            // =========================
            // TANGGAL LAHIR
            // =========================
            $tanggalLahir = null;
            if (!empty($row['tanggal_lahir'])) {
                try {
                    if (is_numeric($row['tanggal_lahir'])) {
                        $tanggalLahir = Date::excelToDateTimeObject($row['tanggal_lahir'])->format('Y-m-d');
                    } else {
                        $tanggalLahir = date('Y-m-d', strtotime($row['tanggal_lahir']));
                    }
                } catch (\Throwable $e) {
                    $tanggalLahir = null;
                }
            }

            $batch[] = [
                'nik'              => $nik,
                'nip'              => $nip,
                'nuptk'            => $nuptk,
                'nama_guru'        => $row['nama'] ?? null,
                'sekolah_id'       => $sekolahId,
                'tempat_lahir'     => $row['tempat_lahir'] ?? null,
                'tanggal_lahir'    => $tanggalLahir,
                'agama'            => $row['agama'] ?? null,
                'email'            => $email,
                'pangkat_golongan' => $row['pangkat_golongan'] ?? null,
                'created_at'       => $now,
                'updated_at'       => $now,
            ];

            if (count($batch) === 500) {
                DB::table('guru')->upsert(
                    $batch,
                    ['email'], // kunci utama anti duplikat
                    [
                        'nik', 'nip', 'nuptk', 'nama_guru', 'sekolah_id',
                        'tempat_lahir', 'tanggal_lahir', 'agama',
                        'pangkat_golongan', 'updated_at'
                    ]
                );
                $batch = [];
            }
        }

        if (!empty($batch)) {
            DB::table('guru')->upsert(
                $batch,
                ['email'],
                [
                    'nik', 'nip', 'nuptk', 'nama_guru', 'sekolah_id',
                    'tempat_lahir', 'tanggal_lahir', 'agama',
                    'pangkat_golongan', 'updated_at'
                ]
            );
        }
    }

    public function chunkSize(): int
    {
        return 500;
    }
}
