<?php

namespace App\Imports;

use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;
use Maatwebsite\Excel\Concerns\ToCollection;
use Maatwebsite\Excel\Concerns\WithHeadingRow;
use Maatwebsite\Excel\Concerns\WithChunkReading;
use PhpOffice\PhpSpreadsheet\Shared\Date;

class PegawaiImport implements ToCollection, WithHeadingRow, WithChunkReading
{
    public function collection(Collection $rows)
    {
        $batch = [];
        $now   = now();

        foreach ($rows as $row) {
            $row = array_change_key_case($row->toArray(), CASE_LOWER);

            // Lewati baris benar-benar kosong
            if (
                empty($row['nama']) &&
                empty($row['nip']) &&
                empty($row['email'])
            ) {
                continue;
            }

            // =========================
            // NORMALISASI DATA
            // =========================
            $nama               = trim($row['nama'] ?? '') ?: null;
            $nip                = trim((string) ($row['nip'] ?? '')) ?: null;
            $gol                = trim($row['gol'] ?? '') ?: null;
            $pangkat_golongan   = trim($row['pangkat_golongan'] ?? '') ?: null;
            $jabatan            = trim($row['jabatan'] ?? '') ?: null;
            $status_kepegawaian = trim($row['status_kepegawaian'] ?? '') ?: null;
            $instansi           = trim($row['instansi'] ?? '') ?: null;
            $email              = trim(strtolower($row['email'] ?? '')) ?: null;
            $telepon            = trim((string) ($row['telepon'] ?? '')) ?: null;
            $alamat             = trim($row['alamat'] ?? '') ?: null;

            // =========================
            // JENIS KELAMIN
            // =========================
            $jk = strtoupper(trim($row['jenis_kelamin'] ?? ''));
            $jenis_kelamin = in_array($jk, ['L', 'P']) ? $jk : null;

            // =========================
            // TANGGAL LAHIR
            // =========================
            $tanggal_lahir = null;
            if (!empty($row['tanggal_lahir'])) {
                try {
                    if (is_numeric($row['tanggal_lahir'])) {
                        $tanggal_lahir = Date::excelToDateTimeObject(
                            $row['tanggal_lahir']
                        )->format('Y-m-d');
                    } else {
                        $tanggal_lahir = date(
                            'Y-m-d',
                            strtotime($row['tanggal_lahir'])
                        );
                    }
                } catch (\Throwable $e) {
                    $tanggal_lahir = null;
                }
            }

            $batch[] = [
                'nama'               => $nama,
                'nip'                => $nip,
                'gol'                => $gol,
                'pangkat_golongan'   => $pangkat_golongan,
                'jabatan'            => $jabatan,
                'status_kepegawaian' => $status_kepegawaian,
                'instansi'           => $instansi,
                'foto'               => null, // diisi manual nanti
                'email'              => $email,
                'telepon'            => $telepon,
                'alamat'             => $alamat,
                'tanggal_lahir'      => $tanggal_lahir,
                'jenis_kelamin'      => $jenis_kelamin,
                'created_at'         => $now,
                'updated_at'         => $now,
            ];

            // Insert per 500 data
            if (count($batch) === 500) {
                $this->upsert($batch);
                $batch = [];
            }
        }

        // Sisa batch
        if (!empty($batch)) {
            $this->upsert($batch);
        }
    }

    protected function upsert(array $batch)
    {
        DB::table('pegawais')->upsert(
            $batch,
            ['email'], // unique key (email)
            [
                'nama',
                'nip',
                'gol',
                'pangkat_golongan',
                'jabatan',
                'status_kepegawaian',
                'instansi',
                'telepon',
                'alamat',
                'tanggal_lahir',
                'jenis_kelamin',
                'updated_at',
            ]
        );
    }

    public function chunkSize(): int
    {
        return 500;
    }
}
